/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @addtogroup PhaseBaseDistEstimate
 * @{
 */

/*! @file
 * This API provides services for processing phases as returned by PDE API and returns distance estimation.
 */

#ifndef _PDE_POSTPROCESSING_H_
#define _PDE_POSTPROCESSING_H_


///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Includes Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <stdint.h>
#include <stdbool.h>
#include "dm_phasebased.h"

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Defines & Macros Section
///////////////////////////////////////////////////////////////////////////////////////////////////

/*! Indicates the maximum number of distance estimates to be calculated by the postprocessing software */
#define LCL_PDE_POSTPROC_MAX_ESTIMATES        (5U)

/*! Describes which filters should be applied to measurement */
#define LCL_PDE_POSTPROC_FILTER_HISTORICAL    (1UL << 0U)  /*!< Historical filter: compare last measurement to previously archived measurements */
#define LCL_PDE_POSTPROC_FILTER_ALL           (0xFFFFFFFFU)

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Typedef Section
///////////////////////////////////////////////////////////////////////////////////////////////////

typedef dm_freq_mask_t pde_postproc_freq_mask_t;

/*! Structure used to return distance estimation and quality metric */
typedef struct pde_estimate_tag
{
    int32_t distance_estimate;   /*!< Estimated distance in Q21.10 format */
    q15_t dqi;                   /*!< Distance quality indicator metric in Q2.14 format */
} pde_estimate_t;

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                Function-like Macros Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Extern Constants Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Extern Variables Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                Function Prototypes Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#if defined(__cplusplus)
extern "C" {
#endif // __cplusplus

/*!
 * Return distance estimation for a given phase vector and frequency mask.
 * The method used is a combination of frequency slope estimation and CDE algorithm.
 *
 * @param [in]  PhaseVector      Pointer to phase vector (one phase per frequency)
 * @param [in]  FreqMask         Pointer to frequency mask (one bit per frequency, 1=valid)
 * @param [in]  NumberFreqs      Number of frequencies in the vector
 * @param [in]  FreqStep         Frequency interval between two entries in the PhaseVector
 * @param [in] ZeroMComp         zero-meters compensation value, Q10 format.
 * @param [in] filterFlags       description of which filters should be applied to the measurement. Affects return value.
 * @param [out] ChosenEstimate   Pointer to output structure, will contain distance in meters
 * 
 * @retval measureIsValid        TRUE if filters did validate the measurement, FALSE if filters detected some inconsistency.
 * @see pde_estimate_t, dm_complex_domain_based_fp
 */
bool PDE_PostprocessingDistanceEstimation(int16_t * PhaseVector, pde_postproc_freq_mask_t * FreqMask, uint16_t NumberFreqs, uint16_t FreqStep,
                                          int32_t ZeroMComp, uint32_t filterFlags, pde_estimate_t * ChosenEstimate);

/*!
 * Function to initialize/reset postprocessing module.
 * Must be called at least once before using PDE.
 * Clear all postprocessing persistant data (history cache, ...)
 */
void PDE_PostprocessingInit(void);

/*!
 * Return IFFT result for last measure, array of q15_t of size N_FFT_MAX.
 * For debugging purpose.
 */
q15_t *PDE_PostprocessingGetCdeData(void);

/*!
 * Return slope-based estimation data for last measure.
 * For debugging purpose.
 */
void PDE_PostprocessingGetSlopeData(pde_estimate_t *estimate);

/*!
 * Returns the CDE distance estimate for the selected index.
 *
 * @param [out] pEstimateResult Pointer to the distance estimate structure for the given index
 * @param [in]  index           Index for the estimate to extract. index must be < LCL_PDE_POSTPROC_MAX_ESTIMATES.
 *
 * @retval Status               0 if executed correctly, 1 if failure occurred.
 */
bool PDE_PostProcessingGetCdeEstimateByIndex (pde_estimate_t* pEstimateResult, uint16_t index);

#if defined(__cplusplus)
}
#endif // __cplusplus


#endif /* _PDE_POSTPROCESSING_H_ */

/*! @} */

///////////////////////////////////////////////////////////////////////////////////////////////////
// EOF
///////////////////////////////////////////////////////////////////////////////////////////////////
